// ExR03Histograms.cc
// 25-Aug-99 Bill Seligman
// This is an example class for booking histograms in GEANT4.

#include "ExR03Histograms.hh"

// Include files for ROOT.
#include "Rtypes.h"
#include "TROOT.h"
#include "TFile.h"
#include "TH1.h"
#include "TH2.h"
#include "TProfile.h"
#include "TNtuple.h"
#include "TRandom.h"

// For the random numbers in the StepFill example.
#include "Randomize.hh"
#include "globals.hh"

// These are needed because they're arguments to the methods in
// the RecorderBase class that we plan to use.  In real life, we
// might use the variables in G4Run or G4Step to 
// histogram or otherwise record values.
#include "G4Run.hh"
#include "G4Step.hh"


ExR03Histograms::ExR03Histograms() {

  // Since we're not creating the TROOT object in the same scope as we're
  // creating our histogram objects, we must declare it to be static.
  // Otherwise, the TROOT object will go away when the constructor is over.

  static TROOT rootBase("simple","Test of histogramming and I/O");
}

ExR03Histograms::~ExR03Histograms() {
}


void ExR03Histograms::RecordBeginOfRun(const G4Run* r)
{
  // Reset the number of times we've called RecordStep.
  countCall = 0;

  // Open a new ROOT file.
  
  hfile = new TFile("hsimple.root","RECREATE","Demo ROOT file with histograms"); 

  // Define our histograms, ntuples, etc.

  myHisto1D = new TH1F("hpx","This is the px distribution",100,-4,4);

  myHisto2D = new TH2F("hpxpy","py vs px",40,-4,4,40,-4,4);

  myProf    = new TProfile("hprof","Profile of pz versus px",100,-4,4,0,20);

  myTuple   = new TNtuple("ntuple","Demo ntuple","px:py:pz:random:i");

}


void ExR03Histograms::RecordEndOfRun(const G4Run* r)
{
  // Save all objects in this file
  hfile->Write();

  // Close the file. Note that this is automatically done when you leave
  // the application.
  hfile->Close();

  // We created the file with "new", so we have to destroy it with "delete":
  // Don't try to delete the histogram/ntuple objects; apparently closing or
  // deleting the file objects already destroys the histograms/ntuples in it.
  delete hfile;
}


void ExR03Histograms::RecordStep(const G4Step*) {

  // Fill histograms randomly
  Float_t px, py, pz;

  gRandom->Rannor(px,py); //px and py will be two gaussian random numbers
  pz = px*px + py*py;
  Float_t random = gRandom->Rndm(1);

  myHisto1D->Fill(px);
  myHisto2D->Fill(px,py);
  myProf->Fill(px,pz);

  myTuple->Fill(px,py,pz,random,countCall++);
}
