// ExR05RootHit.hh
// A simple example hit class to use with ROOT and Geant4.
// 18-Oct-2000 Bill Seligman

#ifndef _EXR05ROOTHIT_H_
#define _EXR05ROOTHIT_H_

// This class can be used with ROOT's dictionary generation, built-in
// I/O, and RTTI (Run-Time Type Identification).  To enable this
// functionality, this class must be derived from class TObject, and
// all its members must also be derived from TObject.
#include "TObject.h"
#include "TVector3.h"

class ExR05RootHit : public TObject 
{
public:

  // Important: A ROOT-compatible class must have a default
  // zero-argument constructor defined, that is, a constructor with
  // either no arguments or one for which all the arguments have
  // default values.
  ExR05RootHit();
  virtual ~ExR05RootHit();

  // Accessor methods.  Note the lack of any public "set" methods.
  // This is a crude form of "locking" the contents of this class;
  // once an hit is created, I don't want any other programs to change
  // its values.  (A real application would have a more sophisticated
  // approach to allow, e.g., a reconstruction program would need to
  // add tracks, momenta, vertices, etc.)
  Double_t GetEnergy() const { return m_energyDeposit; }
  Double_t X() const { return m_position.X(); }
  Double_t Y() const { return m_position.Y(); }
  Double_t Z() const { return m_position.Z(); }
  TVector3 GetPosition() const { return m_position; }
  void Print(Option_t* a_option="") const; // option="verbose" means print run# & event# with hit info

  // The following set functions are protected, not public, because I
  // inherit a Geant4-ROOT "persistency" class from this one.  It is
  // this persistency class that will set the values of the data
  // memebers.
protected:
  void SetEnergy(Double_t e) { m_energyDeposit = e; }
  void SetPosition(Double_t x, Double_t y, Double_t z) { m_position = TVector3(x,y,z); }

private:
  TVector3 m_position;        // position of hit [cm].
  Double_t m_energyDeposit;   // Energy deposited by hit.

public:
  // The following is a macro defined in TObject.h.  The second number
  // is the "version number" of this class definition; if we change
  // the class, we should increment the version number.  This allows
  // ROOT to read files created with several different versions of a
  // class (very handy for reading old files during the course of an
  // experiment's analysis!).
  ClassDef(ExR05RootHit,1) // Simple hit class for example R05
};

#endif // _EXR05ROOTHIT_H_
