//_____________________________________________________________
// ExR05RootEvent
//
// A simple example event class to use with ROOT and Geant4.
// 18-Oct-2000 Bill Seligman
//
// More Geant4+ROOT examples can be found at the 
// begin_html <a href="http://www.nevis.columbia.edu/~atlas/lar/">ATLAS LAr Software at Nevis</a> end_html 
// web page.

// Note that when using ROOT's automatic documentation facility,
// comments at the top of the source file preceeded by
// ""//______________________" will become the class documentation.
// Comments immediately following the opening brace ({) of a member
// description become the member documentation.

#include "ExR05RootEvent.hh"
#include "ExR05RootHit.hh"

#include "TOrdCollection.h"
#include "TString.h"

#include <iostream>

// This class can be used with ROOT's dictionary generation, built-in
// I/O, and RTTI (Run-Time Type Identification).  To enable this
// functionality, this class must be derived from class TObject, all
// its members must also be derived from TObject, and the following
// macro (defined in TObject.h) must be included.
ClassImp(ExR05RootEvent)

ExR05RootEvent::ExR05RootEvent()
{
  // Class constructor.  Creates an empty event.
  m_runNumber = -1;
  m_eventNumber = -1;
  m_hits = 0;
}

ExR05RootEvent::~ExR05RootEvent()
{
  // Class destructor.  Deletes all adopted hits (see AdoptHit).
  if (m_hits != 0)
    {
      m_hits->Delete();
      delete m_hits;
    }
}

Int_t ExR05RootEvent::GetNumberHits() const
{
  // Returns the number of hits associated with this event.
  if (m_hits == 0)
    {
      return 0;
    }
  else
    {
      return m_hits->GetSize();
    }
}

ExR05RootHit* ExR05RootEvent::GetHit(Int_t a_hitNumber) const
{
  // Get a specific hit.  Note that there is no protection against an
  // invalid hit number.

  // The TObject* returned by the "At" method has to be cast into an
  // ExR05RootHit pointer to be returned.
  if (m_hits != 0)
    {
      return (ExR05RootHit*) m_hits->At(a_hitNumber);
    }
  else
    {
      return 0;
    }
}

void ExR05RootEvent::Print(Option_t* a_option) const
{
  // Print the event.  If the option string contains "hits", then all
  // the hits in the event will also be printed.
  if (m_hits != 0)
    {
      TString option = a_option;
      option.ToUpper();

      std::cout << "Run ID=" << m_runNumber << ", Event ID=" << m_eventNumber 
		<< ", Number of Hits=" << GetNumberHits() << std::endl; 

      if (option.Contains("HITS")) 
	{
	  /*
	    // Note that the following command prints all the objects in the
	    // TCollection, that is, all the hits.
	    m_hits->Print(a_option);
	  */

	  // Another approach (illustrates iterators with TCollections):
	  // Note a very big difference between ROOT iterators and STL
	  // iterators: when you create a ROOT iterator, by default it
	  // points to the beginning of the list; each time you invoke
	  // operator() the iterator moves to the next item in the list.
	  Int_t i = 0;
	  TIter next(m_hits); // next is an iterator that points to the beginning of m_hits
	  ExR05RootHit* hit;
	  while ((hit = (ExR05RootHit*) next()))
	    {
	      std::cout << " Hit " << i++ << ": ";
	      hit->Print(a_option);
	    }
	}
    }
}


void ExR05RootEvent::AdoptHit(ExR05RootHit* a_hit)
{
  // Add the pointer at the end of the collection of hits.  As the
  // name of this method implies, we "adopt" the pointer, meaning we
  // take responsibility for deleting it.

  // Create the hit collection if it doesn't already exist.
  if (m_hits == 0) m_hits = new TOrdCollection();

  m_hits->Add(a_hit);
}
