// analysis.C
// An example of analysis code for reading a file based on ROOT Object I/O.
// 22-Nov-2000 Bill Seligman

// Note the #ifdef's that depend on __CINT__.  They are used because
// this code is meant to be either compiled ("make analysis"), or be
// interpreted ("root analysis.C", or type ".x analysis.C" within
// ROOT).
#ifndef __CINT__

#include "ExR05RootEvent.hh"
#include "ExR05RootHit.hh"

#include "TROOT.h"
#include "TRint.h"

#include "TFile.h"
#include "TKey.h"
#include "TOrdCollection.h"
#include "TIterator.h"
#include "TString.h"
#include "TH1.h"
#include "TCanvas.h"

#include <iostream>

// Create the ROOT environment.
TROOT rootBase("analysis","ROOT analysis environment");

int main(int argc, char* argv[])
{
  // Initialize ROOT application.
  TRint *theApp = new TRint("ROOT analysis", &argc, argv);
#else
{
  gROOT->Reset();
  
  // Test if we have to load shared libraries.  For some reason, the
  // physics library is part of the normal ROOT interactive
  // application.
  if (!TClassTable::GetDict("TVector3")) {
    cout << "Loading physics library..." << endl;
    gSystem->Load("$ROOTSYS/lib/libPhysics.so");
  }

  if (!TClassTable::GetDict("ExR05RootEvent")) {
    cout << "Loading ExR05 library..." << endl;
    gSystem->Load("ExR05.so");
  }
#endif

  TFile* f = new TFile("exampleR06.root");

  // If we wanted to get a particular event from the file, we could
  // use:

  // ExR05RootEvent* event = (ExR05RootEvent*)f.Get("ExR05RootEvent;52");

  // for the 52nd event written to the file.  However, the following
  // procedure is more efficient if we want to read through the file
  // sequentially.

  // Reminder: each time we call TIter::operator() (that is, each time
  // we use "nextkey()") we will get a key in the list, and increment
  // nextkey to return the next key.
  TIter nextkey(f->GetListOfKeys());

  // We're going to create a new histogram for each run.
  Int_t saveRun = -1;
  TOrdCollection* histogramList = new TOrdCollection();
  TH1D* hitHistogram;

  // Note that the ROOT methods below (TIter::operator() and
  // TKey::ReadObj) return TObject pointers, and we have to cast them
  // into the correct pointer type for our task.

  // For each event in the file...
  TKey* key;
  while (key = (TKey*)nextkey())
    {
      // Read in an event. 
      ExR05RootEvent* anEvent = (ExR05RootEvent*)key->ReadObj();

      // Create a new histogram for each run.
      if (anEvent->GetRunID() != saveRun)
	{
	  saveRun = anEvent->GetRunID();

	  // Let's make a new histogram for this run.
	  char runText[3];
	  sprintf(runText,"%d",saveRun);
	  TString histName = "zHist" + TString(runText);
	  TString histTitle = "z-distribution of energy deposited by hits for Run " + TString(runText);
	  hitHistogram = new TH1D(histName,histTitle,240,0.,600.);
	  hitHistogram->SetXTitle("z-position [cm]");
	  hitHistogram->SetYTitle("Energy [GeV]");

	  // Add the histogram to our list.
	  histogramList->Add(hitHistogram);
	}

      // For each hit in the event...
      Int_t numHits = anEvent->GetNumberHits();
      for (Int_t j = 0; j < numHits; j++) 
	{
	  // Note how reading the ExR05RootEvent caused its
	  // TOrdCollection of ExR05RootHits to be brought in as well.
	  // All the pointers were correctly set by ROOT's I/O
	  // mechanism.
	  ExR05RootHit* aHit = anEvent->GetHit(j);
	  Stat_t edep = aHit->GetEnergy();
	  Axis_t zpos = aHit->Z();
	  hitHistogram->Fill(zpos,edep);
	}
    }


  // For each histogram we've created, create a Canvas to display it.
  // This is a semi-general piece of histogram display code; all
  // histograms (including the 2- and 3-dimensional ones) derive from
  // the TH1 base class.
  TOrdCollection* canvasList = new TOrdCollection();
  TIter nextHist(histogramList);
  TH1* h;
  Int_t i = 0;
  while (h = (TH1*) nextHist())
    {
      TCanvas* c = new TCanvas(h->GetName(),h->GetTitle(),10+30*i,10+30*i,600,600);
      h->Draw();
      canvasList->Add(c);
      i++;
    }

#ifndef __CINT__
  // Go into interactive mode once we're done.  Initialize intrinsics,
  // build all windows, and enter event loop.
  cout << "Entering interactive mode.  Type .q to exit ROOT." << endl;
  theApp->Run();
#endif

  // Depending on how rigorous we are about memory leaks, and if this
  // was a batch application, we'd want to execute something like the
  // following at the end of the program.

  // canvasList->Delete();
  // delete canvasList;
  // histogramList->Delete();
  // delete histogramList;

  // Note: Be careful not to execute f->Write().  By default, ROOT
  // will try to write all the histograms we've created to the file,
  // and we've accessed it read-only by default.

  return (0);
}
